/************************************************************************
* \file: trace_socket.c
*
* \version: $Id: trace_linux.h,v 1.0 2010/08/04 
*
* Implements IPC using domain sockets
* \component: Gen2 Trace
*
* \author   Diezel, Eckhard (ITUC; ADITG/SWG) [ediezel@de.adit-jv.com]
*           Sakthivelu S.  sakthivelu.s@in.bosch.com  
*
* \copyright: (c) 2003 - 2009 ADIT
*
***********************************************************************/

#include "trace_base.h"


void TRACE_socket_init(TRACE_socket* ds)
{
	if (ds != NULL)
	{
		ds->m_socket = 0;
		ds->m_bound = FALSE;
	}
}

void TRACE_socket_close(TRACE_socket* ds)
{
	if (ds != NULL)
	{
		if (ds->m_socket > 0)
		{
			close(ds->m_socket);
			if (ds->m_bound)
			{
				unlink(ds->m_saun.sun_path);
			}
			ds->m_socket = 0;
			ds->m_bound = FALSE;
		}
	}
}


/*
 * Create a socket
 *
 * \parm ds         Pointer to internal data about this socket
 *                  The same pointer must be passed to all
 *                  TRACE_socket_ functions when dealing with the same
 *                  socket
 *
 * \parm receivetimeout	is currently not used by the calling functions
 * 					It would set a timeout so that a receive socket
 * 					would not block forever when no data arrives
 *
 * \parm adr 		path and name for the socket to be created
 *
 * \parm bindserver	if set to TRUE the socket will be bound to the given adr
 * 					so it can be used with TRACE_socket_receive()
 * 					if set to FALSE the socket remains unbound and can be used
 * 					with TRACE_socket_send()
 *
 * return \li E_OK  if successful
 *                  else error values returned by concrete implementation
 */
ER TRACE_socket_open(TRACE_socket* ds, S32 receivetimeout, 
                          const S8* adr, BOOL bindserver)
{
   ER  ret = E_FAIL;
   S32 len = 0;
   U8  err = 0;
   
   if (ds->m_socket > 0)
   {
	TRACE_socket_close(ds);
   }

   /*
   * Get a socket to work with.  This socket will
   * be in the UNIX domain, and will be a
   * datagram socket.
   */


  ds->m_socket = socket(AF_UNIX, SOCK_DGRAM, 0);

  ds->m_bound = FALSE;

  if (bindserver && ds->m_socket > 0)
  {
      
	  /*
	  * Create the address we will be binding or sending to.
	  *
	  * prevent buffer overflow along the way (sun_path usually is 128 bytes)
	  */
	  ds->m_saun.sun_family = AF_UNIX;
	  ds->m_saun.sun_path[sizeof(ds->m_saun.sun_path)-1] = '\0';
	  strncpy(ds->m_saun.sun_path, (char*)adr, sizeof(ds->m_saun.sun_path)-1);

    /*
     * Try to bind the address to the socket.  We
     * unlink the name first so that the bind won't
     * fail.
     *
     * The third argument indicates the "length" of
     * the structure, not just the length of the
     * socket name.
     */
    unlink((char*)adr);
    len = sizeof(ds->m_saun.sun_family) + strlen(ds->m_saun.sun_path);
    /* PRQA: Lint Message 64: casting from sockaddr_in* to sockaddr* is the standard way of doing */
    /* And hence no change is required */
    /*lint -save -e64 */
    if (bind(ds->m_socket, (const struct sockaddr *)&(ds->m_saun), len) < 0)
    /*lint -restore */
    {
      err = errno;     
      TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "TRACE socket open: bind to adr %s failed [%2x] \n",
              adr,err);
      close(ds->m_socket);
      ds->m_socket = 0;
    }
    else if (receivetimeout > 0)
    {
      struct timeval tv;
      tv.tv_sec = receivetimeout;
      tv.tv_usec = 0;
      if (setsockopt(ds->m_socket, SOL_SOCKET, SO_RCVTIMEO, 
                                    (S8*)&tv,  sizeof tv))
      {
        err = errno;
        TRACE_SYSLOG(TRACE_SYSLOG_ERROR,"TRACE socket open: set sockopt failed [%2x] \n",err);
    	return E_FAIL;
      }
    }

    //change the permision of the socket
    if(chmod((char*)adr,S_IRUSR | S_IWUSR | S_IWGRP | S_IRGRP) == -1)
    { 
       TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Unable to Change mode of socket to 0660\n");
    }
  }
  if  (ds->m_socket > 0)
  {
	  ds->m_bound = bindserver;
	  ret = E_OK;
  }

  return ret;
}


ER TRACE_socket_open_sender(TRACE_socket* ds)
{
	ER ret = E_FAIL;

	if (ds != NULL)
	{
		ret = TRACE_socket_open(ds, 0, (S8*)"", FALSE);
	}

	return ret;
}

ER TRACE_socket_open_receiver(TRACE_socket* ds, const S8* tasksocket, 
                                                  BOOL with_pid)
{
	ER ret                       = E_FAIL;
	pid_t mypid                  = 0;
  S8 mypath[TRACE_FILE_NM_LEN] = {0};

	if (ds != NULL)
	{
		if (with_pid)
		{
			mypid = getpid();
			sprintf((char*)mypath, TRACE_PID_FORMAT, tasksocket, mypid);
			ret = TRACE_socket_open(ds, 0, mypath, TRUE);
		}
		else
		{
			ret = TRACE_socket_open(ds, 0, tasksocket, TRUE);
		}
	}

	return ret;
}


ER TRACE_socket_send(TRACE_socket* ds, S32 topid, const S8* recvname, 
                                                void* buf, S32 len)
{
  ER ret                      = E_FAIL;	
  S32 slen                    = 0;
  S8 topath[TRACE_FILE_NM_LEN]= {0};
  U8 err                      = 0;
	if (ds != NULL)
	{
		/*
		* Create the address we will be binding or sending to.
		*
		* prevent buffer overflow along the way (sun_path usually is 128 bytes)
		*/
		ds->m_saun.sun_family = AF_UNIX;
		ds->m_saun.sun_path[sizeof(ds->m_saun.sun_path)-1] = '\0';
		if (topid == 0)
		{
			strncpy(ds->m_saun.sun_path, (char*)recvname, sizeof(ds->m_saun.sun_path)-1);
		}
		else
		{
			sprintf((char*)topath, TRACE_PID_FORMAT, recvname, topid);
			strncpy(ds->m_saun.sun_path, (char*)topath, sizeof(ds->m_saun.sun_path)-1);
		}
    slen = sizeof(ds->m_saun);

    if (ds->m_socket > 0)
    {
      /* PRQA: Lint Message 64: casting from sockaddr_in* to sockaddr* is the standard way of doing */
      /* And hence no change is required */
      /*lint -save -e64 */
      ret = sendto(ds->m_socket, buf, len, 0, 
        (const struct sockaddr*)&(ds->m_saun), slen);

      /*lint -restore */
	  if(ret < E_OK)
	  {
        err = errno;
        TRACE_SYSLOG(TRACE_SYSLOG_ERROR,"TRACE socket send: sendto failed [%2x]\n",err);
	  }
    }
	}
	return ret;
}

ER TRACE_socket_receive(TRACE_socket* ds, void* buf, S32 len)
{
	ER ret = E_FAIL;
    U8 err = 0;
	
	if (ds != NULL)
	{
	    if ((ds->m_socket > 0) && (ds->m_bound))
	    {
	      ret = recv(ds->m_socket, buf, len, 0);
          if(ret < E_OK)
	      {
            err = errno;
            TRACE_SYSLOG(TRACE_SYSLOG_ERROR,"TRACE socket receive: recv failed [%2x]\n",err);
          }
	    }
	}

	return ret;
}
